{ ROSMAC.INC - Remote Operating System Machine Dependent Routines }
{ Modified for Kaypro and Z-clock by Norman Saunders - 5 Apr 85 }

const
{ Machine specific constants }

  DataPort   = $04;          { Data port }
  StatusPort = $06;          { Status port }
  RatePort   = $00;          { Data rate (bps) port }
  BasePort   = $B0;          { Base port for Z-Clock access}

{ StatusPort commands }

  RESCHN     = $18;          { reset channel }
  RESSTA     = $10;          { reset ext/status }
  WRREG1     = $00;          { value to write to register 1 }
  WRREG3     = $C1;          { 8 bits/char, rx enable }
  WRREG4     = $44;          { 16x, 1 stop bit, no parity }
  DTROFF     = $68;          { dtr off, rts off }
  DTRON      = $EA;          { dtr on, 8 bits/char, tx enable, rts on }
  ONINS      = $30;          { error reset }

{ StatusPort status masks }

  DAV        = $01;          { data available }
  TRDY       = $04;          { transmit buffer empty }
  DCD        = $08;          { data carrier detect }
  PE         = $10;          { parity error }
  OE         = $20;          { overrun error }
  FE         = $40;          { framing error }
  ERR        = $60;          { parity, overrun and framing error }

{ Smartmodem result codes }

  OKAY        = '0';         { Command executed with no errors }
  CONNECT300  = '1';         { Carrier detect at 300 bps }
  RING        = '2';         { Ring signal detected }
  NOCARRIER   = '3';         { Carrier lost or never heard }
  ERROR       = '4';         { Error in command execution }
  CONNECT1200 = '5';         { Carrier detect at 1200 bps }

{ Rate setting commands }

  BD300      = 5;            { 300 bps }
  BD1200     = 7;            { 1200 bps }

function mdcarck: boolean;
{ Check to see if carrier is present }
begin
  port[StatusPort] := RESSTA;
  mdcarck := ((DCD and port[StatusPort]) <> 0)
end;

function mdinprdy: boolean;
{ Check for ready to input from modem }
var
  bt: byte;
begin
  if (DAV and port[StatusPort]) <> 0
    then
      begin
        port[StatusPort] := 1;
        if (ERR and port[StatusPort]) <> 0
          then
            begin
              port[StatusPort] := ONINS;
              bt := port[DataPort];
              mdinprdy := FALSE
            end
          else mdinprdy := TRUE
      end
    else mdinprdy := FALSE
end;

function mdinp: byte;
{ Input a byte from modem - no wait - assumed ready }
begin
  mdinp := port[DataPort]
end;

procedure mdout(b: byte);
{ Output a byte to modem - wait until ready }
begin
  repeat
  until (TRDY and port[StatusPort]) <> 0;
  port[DataPort] := b
end;

procedure mdinit;
{ Initialize the sio channel and the Hayes Smartmodem 1200 }
const
  sio_init: array[1..9] of byte = (RESCHN, 4, WRREG4, 1, WRREG1, 3, WRREG3, 5, DTROFF);
var
  i: integer;
  mdm_attn : string[2];
  mdm_init : string[41];
  bt       : byte;
begin
  for i := 1 to 9 do
    port[StatusPort] := sio_init[i];        { initialize the SIO channel }
  port[StatusPort] := 5;                    { pull DTR high }
  port[StatusPort] := DTRON;
  mdm_attn := 'AT';
  mdm_init := 'ATE0Q0V0M0X1 S0=0 S2=3 S4=255 S5=255';
  port[RatePort] := BD1200;                 {set the 8116 to 1200 baud}
  delay (500);                              {let the modem settle for a bit}
  for i := 1 to 2 do
    begin
      bt := ord(mdm_attn[i]);               {force the modem to 1200 baud}
      mdout(bt)
    end;
  bt := ord(CR);
  mdout(bt);
  delay (2000);                             {wait a sec...}
  for i := 1 to 41 do
    begin
      bt := ord(mdm_init[i]);               {initialize the modem}
      mdout(bt)
    end;
  bt := ord(CR);
  mdout(bt);
  bt := mdinp;                              { clear any previous rings }
  bt := mdinp
end;

function mdring: boolean;
{ Determine if the phone is ringing }
begin
  if mdinprdy
    then mdring := (RING = chr(mdinp))
    else mdring := FALSE
end;

procedure mdhangup;
{ Hangup modem }
var
  i        : integer;
  mdm_hang : string[4];
  bt       : byte;
begin
  repeat
    port[StatusPort] := 5;             { setup to write register 5 }
    port[StatusPort] := DTROFF;        { clear DTR, causing hangup }
    delay(2000);
    port[StatusPort] := 5;
    port[StatusPort] := DTRON;
    if mdcarck
      then
        begin
          mdm_hang := 'ATH0';
          for i := 1 to 3 do
            begin
              bt := ord(ETX);
              mdout(bt)
            end;
          delay(1500);
          for i := 1 to 4 do
            begin
              bt := ord(mdm_hang[i]);
              mdout(bt)
            end;
          bt := ord(CR);
          mdout(bt)
        end;
  until not(mdcarck)
end;

procedure mdans;
{ Detect and set system to rate at which modem answered phone }
var
  mdm_answ : string[3];
  code     : char;
  i        : integer;
  bt       : byte;
begin
  repeat
  until mdinprdy;
  bt := mdinp;
  mdm_answ := 'ATA';
  for i := 1 to 3 do
    begin
      bt := ord(mdm_answ[i]);
      mdout(bt)
    end;
  bt := ord(CR);
  mdout(bt);
  repeat
  until mdinprdy;
  code := chr(mdinp);
  if code = CONNECT1200
    then
      begin
        port[RatePort] := BD1200;
        rate := 0.02075;
        delay(500);
        bt := mdinp;
        bt := mdinp
      end;
  if code = CONNECT300
    then
      begin
        port[RatePort] := BD300;
        rate := 0.083;
        delay(500);
        bt := mdinp;
        bt := mdinp
      end;
  if code = NOCARRIER
    then mdhangup
end;

procedure GetTAD(var t: tad_array);
{ Return a 6 element byte array of the current system time in
  seconds, minutes, hours, day, month, and year. }
  var
    i    : integer;
    Time : array [0..12] of byte;           { Array filled in from Z-Clock }
  begin
    for i := 0 to 12 do                     { This section reads the Z-Clock }
      Time[i] := port[BasePort + i];
    t[0] := Time[1] * 10 + Time[0];
    if Time[5] < 4
      then t[2] := Time[5] * 10 + Time[4]
    else if Time[5] >= 8
      then t[2] := (Time[5] - 8) * 10 + Time[4]
      else t[2] := (Time[5] - 4) * 10 + Time[4];
    t[1] := Time[3] * 10 + Time[2];
    if Time[8] >= 4
      then Time[8] := Time[8] - 4;
    t[3] := Time[8] * 10 + Time[7];
    t[4] := Time[10] * 10 + Time[9];
    t[5] := Time[12] * 10 + Time[11]
  end;

procedure SetTAD(var t: tad_array);
{ Set the system time using  a 6 element byte array }
  var
    i    : integer;
    Time : array [0..12] of byte;           { Array filled in from Z-Clock }
  begin
    Time[0] := 0;
    Time[1] := 0;
    Time[2] := t[1] mod 10;
    Time[3] := t[1] div 10;
    Time[4] := t[2] mod 10;
    Time[5] := t[2] div 10;
    Time[6] := 0;                           { Day of week computed in ROS }
    Time[7] := t[3] mod 10;
    Time[8] := t[3] div 10;
    Time[9] := t[4] mod 10;
    Time[10]:= t[4] div 10;
    Time[11]:= t[5] mod 10;
    Time[12]:= t[5] div 10;
    For i := 0 to 12 do
      port[BasePort + i] := Time[i]
  end;

